﻿/*
 MIT License

 Copyright (c) 2025 PROCYON STUDIO CO., LTD.
 Copyright (c) 2025 KIRA Ryouta

 Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

using System;
using System.Buffers.Binary;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Linq;
using System.Net;
using System.Net.Http.Headers;
using System.Net.NetworkInformation;
using System.Runtime.InteropServices;
using System.Text;

namespace HandyHarp2Client
{
    using System;
    using System.Collections.Generic;
    using System.Collections.Specialized;
    using System.Net.NetworkInformation;
    using System.Text;

    public sealed class NetworkService
    {
        internal NetworkService()
        { }

        public string? Name { get; internal set; }
        public string? Type { get; internal set; }
        public string? Host { get; internal set; }
        public IPAddress[]? IpAddresses { get; internal set; }
        public ushort? Port { get; internal set; }
    }

    public class ServiceDiscoveredEventArgs : EventArgs
    {
        internal ServiceDiscoveredEventArgs(NetworkService service)
        {
            Service = service;
        }
        public NetworkService Service { get; }
    }

    internal class ServiceBrowser : IDisposable
    {
        public bool Running { get; protected set; }

        private const int DNS_REQUEST_PENDING = 0x2522;

        private DNS_SERVICE_CANCEL? cancelHandle;

        private DNS_SERVICE_BROWSE_REQUEST? request;

        public event EventHandler<ServiceDiscoveredEventArgs>? ServiceDiscovered;

        public void Start(string queryName)
        {
            StartInternal(queryName, 0);
        }

        public void Start(string queryName, NetworkInterface networkInterface)
        {
            if (!(networkInterface.Supports(NetworkInterfaceComponent.IPv4) ||
                  networkInterface.Supports(NetworkInterfaceComponent.IPv6)))
            {
                throw new ArgumentException("Network interface does not support IP.");
            }

            var props = networkInterface.GetIPProperties().GetIPv4Properties();
            StartInternal(queryName, (uint)props.Index);
        }

        private void StartInternal(string queryName, uint ifIndex)
        {
            if (Running) throw new InvalidOperationException("The service browser is already running.");
            var cbd = new DNS_QUERY_COMPLETION_ROUTINE(BrowseCallback);
            request = new DNS_SERVICE_BROWSE_REQUEST
            {
                InterfaceIndex = ifIndex,
                pQueryContext = IntPtr.Zero,
                QueryName = queryName,
                pBrowseCallbackV2 = cbd
            };
            cancelHandle = new DNS_SERVICE_CANCEL();
            var result = DnsMethods.DnsServiceBrowse(request, cancelHandle);
            if (result != DNS_REQUEST_PENDING)
                throw new Win32Exception(result);
            Running = true;
        }

        private void NotifyOfNewServiceFromRecords(IntPtr recordPtr)
        {
            var service = new NetworkService();
            var ipAddresses = new List<IPAddress>();
            var srvRecords = new List<DNS_SRV_DATA>();
            var txtRecords = new List<string>();

            while (recordPtr != IntPtr.Zero)
            {
                var record = Marshal.PtrToStructure<DNS_RECORD>(recordPtr);
                if (record == null) break;
                switch (record.wType)
                {
                    case DnsRecordType.DNS_TYPE_A:
                        var addr = new IPAddress(record.Data.A.IpAddress);
                        ipAddresses.Add(addr);
                        break;

                    case DnsRecordType.DNS_TYPE_PTR:
                        var nameHost = Marshal.PtrToStringUni(record.Data.PTR.pNameHost);
                        if (nameHost == null) break;
                        var dotIndex = nameHost.IndexOf('.');
                        if (dotIndex == -1) break;
                        service.Name = nameHost.Substring(0, dotIndex);
                        service.Type = nameHost.Substring(dotIndex + 1);
                        break;

                    case DnsRecordType.DNS_TYPE_AAAA:
                        var bytes = new Span<byte>(new byte[sizeof(ulong) * 2]);
                        BinaryPrimitives.WriteUInt64BigEndian(bytes, record.Data.AAAA.IP6Addr1);
                        BinaryPrimitives.WriteUInt64BigEndian(bytes, record.Data.AAAA.IP6Addr2);
                        ipAddresses.Add(new IPAddress(bytes));
                        break;

                    case DnsRecordType.DNS_TYPE_SRV:
                        srvRecords.Add(record.Data.SRV);
                        break;

                    case DnsRecordType.DNS_TYPE_TEXT:
                        // DNS_TXT_DATA is a flexible last member struct
                        var dataMem = recordPtr + Marshal.SizeOf<DNS_RECORD_NO_DATA_STRUCT>() +
                                      (sizeof(uint) * 2); // 64-bit packing?
                        for (var i = 0; i < record.Data.TXT.dwStringCount; i++)
                        {
                            var strPtr = Marshal.ReadIntPtr(dataMem + (i * IntPtr.Size));
                            var entry = Marshal.PtrToStringUni(strPtr);
                            if (entry != null)
                            {
                                txtRecords.Add(entry);
                            }
                        }
                        break;
                }
                recordPtr = record.pNext;
            }

            service.IpAddresses = ipAddresses.ToArray();
            srvRecords.Sort((a, b) =>
            {
                if (a.wPriority != b.wPriority) return a.wPriority - b.wPriority;
                return b.wWeight - a.wWeight;
            });
            var bestSrvRecord = srvRecords.FirstOrDefault();
            service.Host = Marshal.PtrToStringUni(bestSrvRecord.pNameTarget);
            service.Port = bestSrvRecord.wPort;
            OnServiceDiscovered(new ServiceDiscoveredEventArgs(service));
        }

        private void BrowseCallback(IntPtr pQueryContext, DNS_QUERY_RESULT pQueryResults)
        {
            var recordPtr = pQueryResults.pQueryRecords;
            try
            {
                NotifyOfNewServiceFromRecords(recordPtr);
            }
            finally
            {
                DnsMethods.DnsFree(recordPtr, DNS_FREE_TYPE.DnsFreeRecordList);
            }
        }

        public void Stop()
        {
            if (cancelHandle == null) return;
            DnsMethods.DnsServiceBrowseCancel(cancelHandle);
            cancelHandle = null;
        }

        protected void OnServiceDiscovered(ServiceDiscoveredEventArgs eventArgs)
        {
            ServiceDiscovered?.Invoke(this, eventArgs);
        }

        private void ReleaseUnmanagedResources()
        {
            if (cancelHandle != null) Stop();
        }

        public void Dispose()
        {
            ReleaseUnmanagedResources();
            GC.SuppressFinalize(this);
        }

        ~ServiceBrowser()
        {
            ReleaseUnmanagedResources();
        }
    }
}